<?php
/**
 * importar_productos_orvet_csv.php
 *
 * Importa un CSV (generado desde PRODUCTOS_ORVET.XLS) e inserta:
 *  - categorias  (desde columna "categoria")
 *  - modelos     (desde columna "modelo" = proveedor/laboratorio)
 *  - productos   (1 registro por fila)
 *  - equivalencias (1 por producto, por sucursal)
 *  - costos      (1 por producto, por almacén, fecha y compra_id)
 *
 * CSV esperado (UTF-8):
 * proveedor,modelo,linea,categoria,nombre,costo
 *
 * Seguridad:
 *  - Requiere token por GET o por JSON POST.
 *
 * Ejemplo POST JSON:
 * {
 *   "token":"TU_TOKEN",
 *   "archivo":"productos_orvet_import_fmt_costo0.csv",
 *   "empresa_id":1,
 *   "sucursal_id":1,
 *   "almacen_id":1,
 *   "marca_id_default":1,
 *   "unidad_id":1,
 *   "fraccion_id":1,
 *   "capacidad":1,
 *   "afecto_igv":true,
 *   "porcentaje_igv":18,
 *   "fecha":"2026-01-07",
 *   "doc":"IMPORT-ORVET",
 *   "compra_id":0,
 *   "equiv_nombre":"UND",
 *   "equiv_abrev":"UND",
 *   "equiv_precio_venta":"costo"   // "costo" ó un número (ej. 0)
 * }
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/importar_productos_orvet_csv_error.log');
header("Content-Type: application/json; charset=utf-8");

include_once __DIR__ . '/../conexion.php'; // ajusta ruta si tu estructura difiere

function json_out($ok, $msg, $extra = []) {
    http_response_code($ok ? 200 : 400);
    echo json_encode(array_merge([
        "ESTADO" => $ok ? "1" : "0",
        "MENSAJE" => $msg
    ], $extra), JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

function read_input() {
    $raw = file_get_contents("php://input");
    $data = [];
    if ($raw) {
        $tmp = json_decode($raw, true);
        if (is_array($tmp)) $data = $tmp;
    }
    // también permite GET/POST form
    foreach (["token","archivo","empresa_id","sucursal_id","almacen_id","marca_id_default","unidad_id","fraccion_id","capacidad","afecto_igv","porcentaje_igv","fecha","doc","compra_id","equiv_nombre","equiv_abrev","equiv_precio_venta"] as $k) {
        if (!isset($data[$k]) && isset($_REQUEST[$k])) $data[$k] = $_REQUEST[$k];
    }
    return $data;
}

function boolval_loose($v, $default=false) {
    if ($v === null) return $default;
    if (is_bool($v)) return $v;
    $s = strtolower(trim((string)$v));
    if ($s === "1" || $s === "true" || $s === "t" || $s === "si" || $s === "sí" || $s === "y") return true;
    if ($s === "0" || $s === "false" || $s === "f" || $s === "no" || $s === "n") return false;
    return $default;
}

function norm_text($s, $maxLen=null) {
    $s = trim((string)$s);
    $s = preg_replace('/\s+/', ' ', $s);
    // mantener tildes; solo normaliza espacios
    if ($maxLen !== null && mb_strlen($s, 'UTF-8') > $maxLen) {
        $s = mb_substr($s, 0, $maxLen, 'UTF-8');
    }
    return $s;
}

$data = read_input();

// === CONFIG SEGURIDAD ===
$TOKEN = "CAMBIA_ESTE_TOKEN"; // <-- pon tu token real aquí
if (!isset($data["token"]) || $data["token"] !== $TOKEN) {
    json_out(false, "Acceso no autorizado (token inválido).");
}

// === Parámetros (con defaults) ===
$archivo = $data["archivo"] ?? "productos_orvet_import_fmt_costo0.csv";
$empresa_id = (int)($data["empresa_id"] ?? 1);
$sucursal_id = (int)($data["sucursal_id"] ?? 1);
$almacen_id = (int)($data["almacen_id"] ?? 1);

$marca_id_default = (int)($data["marca_id_default"] ?? 1);
$unidad_id = (int)($data["unidad_id"] ?? 7);
$fraccion_id = (int)($data["fraccion_id"] ?? 7);
$capacidad = (float)($data["capacidad"] ?? 1);

$afecto_igv = boolval_loose($data["afecto_igv"] ?? true, true);
$porcentaje_igv = (float)($data["porcentaje_igv"] ?? 18);

$fecha = $data["fecha"] ?? date("Y-m-d");
$doc = $data["doc"] ?? "IMPORT-ORVET";
$compra_id = (int)($data["compra_id"] ?? 0);

$equiv_nombre = norm_text($data["equiv_nombre"] ?? "UND", 30);
$equiv_abrev  = norm_text($data["equiv_abrev"] ?? "NIU", 4);
$equiv_precio_venta = $data["equiv_precio_venta"] ?? "costo"; // "costo" o número

// === Ruta del CSV ===
$csvPath = __DIR__ . "/" . basename($archivo);
if (!file_exists($csvPath)) {
    json_out(false, "No se encontró el CSV en el servidor.", ["ruta_esperada" => $csvPath]);
}

try {
    // Asegura modo excepción
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Statements preparados (se reusan)
    $stCatFind = $pdo->prepare("SELECT id FROM categorias WHERE empresa_id = :empresa_id AND upper(nombre) = upper(:nombre) LIMIT 1");
    $stCatIns  = $pdo->prepare("INSERT INTO categorias(nombre, offsystem, activo, empresa_id) VALUES (:nombre, false, true, :empresa_id) RETURNING id");

    $stModFind = $pdo->prepare("SELECT id FROM modelos WHERE empresa_id = :empresa_id AND upper(nombre) = upper(:nombre) LIMIT 1");
    $stModIns  = $pdo->prepare("INSERT INTO modelos(nombre, offsystem, activo, empresa_id) VALUES (:nombre, false, true, :empresa_id) RETURNING id");

    // Insert/Update producto con protección por empresa_id (ver WHERE del ON CONFLICT)
    $stProdUpsert = $pdo->prepare("
        INSERT INTO productos(
            codigo_producto,
            nombre, descripcion,
            codigo_barras, codigo_externo, codigo_sunat,
            afecto_igv, porcentaje_igv,
            pcompra_ant, fcompra_ant, doccompra_ant,
            ult_pcompra, ult_fcompra, ult_doccompra,
            peso, stock_minimo,
            pprecio_menor, pprecio_mayor, pprecio_dist,
            precmenor_und, precmenor_fra,
            precmayor_und, precmayor_fra,
            precdist_und, precdist_fra,
            importe_icbp,
            contenido, informacion, path_foto,
            usa_serie, offsystem, activo,
            unidad_id, capacidad, fraccion_id,
            categoria_id, marca_id, modelo_id,
            empresa_id,
            precio_stock, isc, fise
        ) VALUES (
            NULL,
            :nombre, :descripcion,
            NULL, NULL, NULL,
            :afecto_igv, :porcentaje_igv,
            :costo, :fecha, :doc,
            :costo, :fecha, :doc,
            0, 0,
            0, 0, 0,
            0, 0,
            0, 0,
            0, 0,
            0,
            '-', '-', NULL,
            false, false, true,
            :unidad_id, :capacidad, :fraccion_id,
            :categoria_id, :marca_id, :modelo_id,
            :empresa_id,
            0, 0, 0
        )
        ON CONFLICT (descripcion)
        DO UPDATE SET
            categoria_id = EXCLUDED.categoria_id,
            modelo_id = EXCLUDED.modelo_id,
            ult_pcompra = EXCLUDED.ult_pcompra,
            ult_fcompra = EXCLUDED.ult_fcompra,
            ult_doccompra = EXCLUDED.ult_doccompra
        WHERE productos.empresa_id = EXCLUDED.empresa_id
        RETURNING id, codigo_producto
    ");

    $stProdGetByDesc = $pdo->prepare("SELECT id, codigo_producto, empresa_id FROM productos WHERE descripcion = :descripcion LIMIT 1");

    $stEqFind = $pdo->prepare("SELECT id FROM equivalencias WHERE producto_id=:producto_id AND empresa_id=:empresa_id AND sucursal_id=:sucursal_id AND abrev_precio=:abrev LIMIT 1");
    $stEqIns  = $pdo->prepare("
        INSERT INTO equivalencias(
            producto_id, nombre, capacidad_precio, abrev_precio, abrev_universal,
            precio_venta, empresa_id, sucursal_id, iud, codigo_producto
        ) VALUES (
            :producto_id, :nombre, :capacidad_precio, :abrev_precio, :abrev_universal,
            :precio_venta, :empresa_id, :sucursal_id, '', :codigo_producto
        )
        RETURNING id
    ");

    $stCostIns = $pdo->prepare("
        INSERT INTO costos(almacen_id, producto_id, costo, costo_promedio, flete, estiba, fecha, compra_id)
        VALUES (:almacen_id, :producto_id, :costo, :costo_promedio, 0, 0, :fecha, :compra_id)
        ON CONFLICT (almacen_id, producto_id, fecha, compra_id) DO NOTHING
    ");

    $pdo->beginTransaction();

    $handle = fopen($csvPath, "r");
    if ($handle === false) {
        throw new RuntimeException("No se pudo abrir el CSV.");
    }

    $header = fgetcsv($handle); // lee cabecera
    if (!$header) {
        throw new RuntimeException("CSV vacío o sin cabecera.");
    }

    // Mapeo de columnas por nombre (robusto)
    $cols = array_map(function($h){ return strtolower(trim($h)); }, $header);
    $ix = function($name) use ($cols) {
        $pos = array_search(strtolower($name), $cols, true);
        return $pos === false ? -1 : $pos;
    };

    $iProveedor = $ix("proveedor");
    $iModelo    = $ix("modelo");
    $iLinea     = $ix("linea");
    $iCategoria = $ix("categoria");
    $iNombre    = $ix("nombre");
    $iCosto     = $ix("costo");

    if ($iNombre < 0 || $iCosto < 0 || $iCategoria < 0 || $iModelo < 0) {
        throw new RuntimeException("Cabecera CSV no coincide. Se espera: proveedor,modelo,linea,categoria,nombre,costo");
    }

    $count = [
        "filas_csv" => 0,
        "categorias_creadas" => 0,
        "modelos_creados" => 0,
        "productos_insertados_o_actualizados" => 0,
        "equivalencias_creadas" => 0,
        "costos_insertados" => 0
    ];

    $cacheCat = []; // nombre -> id
    $cacheMod = []; // nombre -> id

    while (($row = fgetcsv($handle)) !== false) {
        $count["filas_csv"]++;

        $nombre = norm_text($row[$iNombre] ?? "", 500);
        if ($nombre === "") continue;

        $categoria = norm_text($row[$iCategoria] ?? "", 30);
        $modelo    = norm_text($row[$iModelo] ?? "", 30);
        $costoRaw  = trim((string)($row[$iCosto] ?? "0"));
        $costoRaw  = str_replace(",", ".", $costoRaw);
        $costo     = (float)$costoRaw;

        if ($categoria === "" || $modelo === "") {
            throw new RuntimeException("Fila {$count["filas_csv"]}: categoria/modelo vacío. Revisa el CSV.");
        }

        // === categoria ===
        if (isset($cacheCat[$categoria])) {
            $categoria_id = $cacheCat[$categoria];
        } else {
            $stCatFind->execute(["empresa_id"=>$empresa_id, "nombre"=>$categoria]);
            $categoria_id = $stCatFind->fetchColumn();
            if (!$categoria_id) {
                $stCatIns->execute(["nombre"=>$categoria, "empresa_id"=>$empresa_id]);
                $categoria_id = $stCatIns->fetchColumn();
                $count["categorias_creadas"]++;
            }
            $cacheCat[$categoria] = (int)$categoria_id;
        }

        // === modelo (proveedor/laboratorio) ===
        if (isset($cacheMod[$modelo])) {
            $modelo_id = $cacheMod[$modelo];
        } else {
            $stModFind->execute(["empresa_id"=>$empresa_id, "nombre"=>$modelo]);
            $modelo_id = $stModFind->fetchColumn();
            if (!$modelo_id) {
                $stModIns->execute(["nombre"=>$modelo, "empresa_id"=>$empresa_id]);
                $modelo_id = $stModIns->fetchColumn();
                $count["modelos_creados"]++;
            }
            $cacheMod[$modelo] = (int)$modelo_id;
        }

        // === producto ===
        $stProdUpsert->execute([
            "nombre" => $nombre,
            "descripcion" => $nombre,
            "empresa_id" => $empresa_id,
            "categoria_id" => (int)$categoria_id,
            "marca_id" => $marca_id_default,
            "modelo_id" => (int)$modelo_id,
            "unidad_id" => $unidad_id,
            "fraccion_id" => $fraccion_id,
            "capacidad" => $capacidad,
            "afecto_igv" => $afecto_igv ? "t" : "f",
            "porcentaje_igv" => $porcentaje_igv,
            "costo" => $costo,
            "fecha" => $fecha,
            "doc" => $doc
        ]);

        $prod = $stProdUpsert->fetch(PDO::FETCH_ASSOC);
        if (!$prod) {
            // Puede pasar si hay conflicto por descripcion con otra empresa (WHERE del ON CONFLICT evita update)
            $stProdGetByDesc->execute(["descripcion"=>$nombre]);
            $p = $stProdGetByDesc->fetch(PDO::FETCH_ASSOC);
            if (!$p) {
                throw new RuntimeException("No se pudo insertar ni recuperar el producto por descripcion: {$nombre}");
            }
            if ((int)$p["empresa_id"] !== $empresa_id) {
                throw new RuntimeException("Conflicto UNIQUE(descripcion) con otra empresa (empresa_id={$p["empresa_id"]}) para: {$nombre}");
            }
            $producto_id = (int)$p["id"];
            $codigo_producto = (string)$p["codigo_producto"];
        } else {
            $producto_id = (int)$prod["id"];
            $codigo_producto = (string)$prod["codigo_producto"];
        }
        $count["productos_insertados_o_actualizados"]++;

        // === equivalencia (1 por sucursal) ===
        $stEqFind->execute([
            "producto_id"=>$producto_id,
            "empresa_id"=>$empresa_id,
            "sucursal_id"=>$sucursal_id,
            "abrev"=>$equiv_abrev
        ]);
        $eq_id = $stEqFind->fetchColumn();
        if (!$eq_id) {
            $precio_venta = 0.0;
            if (is_string($equiv_precio_venta) && strtolower(trim($equiv_precio_venta)) === "costo") {
                $precio_venta = $costo;
            } else {
                $precio_venta = (float)str_replace(",", ".", (string)$equiv_precio_venta);
            }

            $stEqIns->execute([
                "producto_id"=>$producto_id,
                "nombre"=>$equiv_nombre,
                "capacidad_precio"=>1,
                "abrev_precio"=>$equiv_abrev,
                "abrev_universal"=>$equiv_abrev,
                "precio_venta"=>$precio_venta,
                "empresa_id"=>$empresa_id,
                "sucursal_id"=>$sucursal_id,
                "codigo_producto"=>$codigo_producto
            ]);
            $count["equivalencias_creadas"]++;
        }

        // === costos ===
        $stCostIns->execute([
            "almacen_id"=>$almacen_id,
            "producto_id"=>$producto_id,
            "costo"=>$costo,
            "costo_promedio"=>$costo,
            "fecha"=>$fecha,
            "compra_id"=>$compra_id
        ]);
        $count["costos_insertados"] += $stCostIns->rowCount();
    }

    fclose($handle);
    $pdo->commit();

    json_out(true, "Importación completada.", [
        "PARAMS" => [
            "archivo"=>$archivo,
            "empresa_id"=>$empresa_id,
            "sucursal_id"=>$sucursal_id,
            "almacen_id"=>$almacen_id,
            "marca_id_default"=>$marca_id_default,
            "unidad_id"=>$unidad_id,
            "fraccion_id"=>$fraccion_id,
            "capacidad"=>$capacidad,
            "afecto_igv"=>$afecto_igv,
            "porcentaje_igv"=>$porcentaje_igv,
            "fecha"=>$fecha,
            "doc"=>$doc,
            "compra_id"=>$compra_id,
            "equiv_nombre"=>$equiv_nombre,
            "equiv_abrev"=>$equiv_abrev,
            "equiv_precio_venta"=>$equiv_precio_venta
        ],
        "RESUMEN" => $count
    ]);

} catch (Throwable $e) {
    if (isset($pdo) && $pdo instanceof PDO && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    json_out(false, "Error en importación.", [
        "ERROR" => [
            "tipo" => get_class($e),
            "mensaje" => $e->getMessage(),
            "linea" => $e->getLine()
        ]
    ]);
}
